/*
 * Decompiled with CFR 0.152.
 */
package org.apache.flink.table.client;

import java.io.Closeable;
import java.net.InetSocketAddress;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.Arrays;
import java.util.function.Supplier;
import javax.annotation.Nullable;
import org.apache.commons.lang3.SystemUtils;
import org.apache.flink.annotation.VisibleForTesting;
import org.apache.flink.configuration.Configuration;
import org.apache.flink.table.client.SqlClientException;
import org.apache.flink.table.client.cli.CliClient;
import org.apache.flink.table.client.cli.CliOptions;
import org.apache.flink.table.client.cli.CliOptionsParser;
import org.apache.flink.table.client.cli.CliUtils;
import org.apache.flink.table.client.gateway.DefaultContextUtils;
import org.apache.flink.table.client.gateway.Executor;
import org.apache.flink.table.client.gateway.SingleSessionManager;
import org.apache.flink.table.gateway.SqlGateway;
import org.apache.flink.table.gateway.api.endpoint.SqlGatewayEndpointFactoryUtils;
import org.apache.flink.table.gateway.rest.util.SqlGatewayRestOptions;
import org.apache.flink.table.gateway.service.context.DefaultContext;
import org.apache.flink.table.gateway.service.session.SessionManager;
import org.apache.flink.util.NetUtils;
import org.jline.terminal.Terminal;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class SqlClient {
    private static final Logger LOG = LoggerFactory.getLogger(SqlClient.class);
    private final boolean isGatewayMode;
    private final CliOptions options;
    private final Supplier<Terminal> terminalFactory;
    public static final String MODE_EMBEDDED = "embedded";
    public static final String MODE_GATEWAY = "gateway";
    public static final String MODE_NONE = "";

    public SqlClient(boolean isGatewayMode, CliOptions options, Supplier<Terminal> terminalFactory) {
        this.isGatewayMode = isGatewayMode;
        this.options = options;
        this.terminalFactory = terminalFactory;
    }

    private void start() {
        if (this.isGatewayMode) {
            CliOptions.GatewayCliOptions gatewayCliOptions = (CliOptions.GatewayCliOptions)this.options;
            try (Executor executor = Executor.create(DefaultContextUtils.buildDefaultContext(gatewayCliOptions), gatewayCliOptions.getGatewayAddress().orElseThrow(() -> new SqlClientException("Please specify the address of the SQL Gateway with command line option '-e,--endpoint <SQL Gateway address>' in the gateway mode.")), this.options.getSessionId());){
                Runtime.getRuntime().addShutdownHook(new ShutdownThread(executor));
                this.openCli(executor);
            }
        }
        DefaultContext defaultContext = DefaultContextUtils.buildDefaultContext((CliOptions.EmbeddedCliOptions)this.options);
        try (EmbeddedGateway embeddedGateway = EmbeddedGateway.create(defaultContext);
             Executor executor = Executor.create(defaultContext, InetSocketAddress.createUnresolved(embeddedGateway.getAddress(), embeddedGateway.getPort()), this.options.getSessionId());){
            Runtime.getRuntime().addShutdownHook(new ShutdownThread(executor, embeddedGateway));
            this.openCli(executor);
        }
    }

    private void openCli(Executor executor) {
        Path historyFilePath = this.options.getHistoryFilePath() != null ? Paths.get(this.options.getHistoryFilePath(), new String[0]) : Paths.get(System.getProperty("user.home"), SystemUtils.IS_OS_WINDOWS ? "flink-sql-history" : ".flink-sql-history");
        boolean hasSqlFile = this.options.getSqlFile() != null;
        try (CliClient cli = new CliClient(this.terminalFactory, executor, historyFilePath);){
            if (this.options.getInitFile() != null) {
                if (CliUtils.isApplicationMode(executor.getSessionConfig())) {
                    throw new SqlClientException("Sql Client doesn't support to run init files when deploying script into cluster.");
                }
                boolean success = cli.executeInitialization(this.options.getInitFile());
                if (!success) {
                    System.out.println(String.format("Failed to initialize from sql script: %s. Please refer to the LOG for detailed error messages.", this.options.getInitFile()));
                    return;
                }
                System.out.println(String.format("Successfully initialized from sql script: %s", this.options.getInitFile()));
            }
            if (!hasSqlFile) {
                cli.executeInInteractiveMode();
            } else {
                cli.executeInNonInteractiveMode(this.options.getSqlFile());
            }
        }
    }

    public static void main(String[] args) {
        SqlClient.startClient(args, CliClient.DEFAULT_TERMINAL_FACTORY);
    }

    @VisibleForTesting
    protected static void startClient(String[] args, Supplier<Terminal> terminalFactory) {
        CliOptions options;
        String[] modeArgs;
        String mode;
        if (args.length < 1 || args[0].startsWith("-")) {
            mode = MODE_NONE;
            modeArgs = args;
        } else {
            mode = args[0];
            modeArgs = Arrays.copyOfRange(args, 1, args.length);
        }
        switch (mode) {
            case "embedded": {
                options = CliOptionsParser.parseEmbeddedModeClient(modeArgs);
                if (!options.isPrintHelp()) break;
                CliOptionsParser.printHelpEmbeddedModeClient(terminalFactory.get().writer());
                return;
            }
            case "gateway": {
                options = CliOptionsParser.parseGatewayModeClient(modeArgs);
                if (!options.isPrintHelp()) break;
                CliOptionsParser.printHelpGatewayModeClient(terminalFactory.get().writer());
                return;
            }
            case "": {
                options = CliOptionsParser.parseEmbeddedModeClient(modeArgs);
                if (!options.isPrintHelp()) break;
                CliOptionsParser.printHelpClient(terminalFactory.get().writer());
                return;
            }
            default: {
                CliOptionsParser.printHelpClient(terminalFactory.get().writer());
                return;
            }
        }
        try {
            SqlClient client = new SqlClient(mode.equals(MODE_GATEWAY), options, terminalFactory);
            client.start();
        }
        catch (SqlClientException e) {
            System.out.println();
            System.out.println();
            LOG.error("SQL Client must stop.", (Throwable)e);
            throw e;
        }
        catch (Throwable t) {
            System.out.println();
            System.out.println();
            LOG.error("SQL Client must stop. Unexpected exception. This is a bug. Please consider filing an issue.", t);
            throw new SqlClientException("Unexpected exception. This is a bug. Please consider filing an issue.", t);
        }
    }

    private static class ShutdownThread
    extends Thread {
        private final Executor executor;
        @Nullable
        private final EmbeddedGateway gateway;

        private ShutdownThread(Executor executor) {
            this(executor, null);
        }

        public ShutdownThread(Executor executor, @Nullable EmbeddedGateway gateway) {
            this.executor = executor;
            this.gateway = gateway;
        }

        @Override
        public void run() {
            System.out.println("\nShutting down the session...");
            this.executor.close();
            if (this.gateway != null) {
                this.gateway.close();
            }
            System.out.println("done.");
        }
    }

    private static class EmbeddedGateway
    implements Closeable {
        private static final String ADDRESS = "localhost";
        private final NetUtils.Port port;
        private final SqlGateway sqlGateway;

        public static EmbeddedGateway create(DefaultContext defaultContext) {
            NetUtils.Port port = NetUtils.getAvailablePort();
            Configuration defaultConfig = defaultContext.getFlinkConfig();
            Configuration restConfig = new Configuration();
            restConfig.set(SqlGatewayRestOptions.ADDRESS, (Object)ADDRESS);
            restConfig.set(SqlGatewayRestOptions.PORT, (Object)port.getPort());
            defaultConfig.addAll(restConfig, SqlGatewayEndpointFactoryUtils.getSqlGatewayOptionPrefix((String)"rest"));
            SqlGateway sqlGateway = new SqlGateway(defaultConfig, (SessionManager)new SingleSessionManager(defaultContext));
            try {
                sqlGateway.start();
                LOG.info("Start embedded gateway on port {}", (Object)port.getPort());
            }
            catch (Throwable t) {
                EmbeddedGateway.closePort(port);
                throw new SqlClientException("Failed to start the embedded sql-gateway.", t);
            }
            return new EmbeddedGateway(sqlGateway, port);
        }

        private EmbeddedGateway(SqlGateway sqlGateway, NetUtils.Port port) {
            this.sqlGateway = sqlGateway;
            this.port = port;
        }

        String getAddress() {
            return ADDRESS;
        }

        int getPort() {
            return this.port.getPort();
        }

        @Override
        public void close() {
            this.sqlGateway.stop();
            EmbeddedGateway.closePort(this.port);
        }

        private static void closePort(NetUtils.Port port) {
            try {
                port.close();
            }
            catch (Exception exception) {
                // empty catch block
            }
        }
    }
}

