/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.dubbo.rpc.protocol.dubbo;

import org.apache.dubbo.common.URL;
import org.apache.dubbo.rpc.AppResponse;
import org.apache.dubbo.rpc.Invoker;
import org.apache.dubbo.rpc.Result;
import org.apache.dubbo.rpc.RpcException;
import org.apache.dubbo.rpc.RpcInvocation;
import org.apache.dubbo.rpc.cluster.filter.ClusterFilter;
import org.apache.dubbo.rpc.protocol.dubbo.filter.FutureFilter;
import org.apache.dubbo.rpc.support.DemoService;

import org.junit.jupiter.api.Assertions;
import org.junit.jupiter.api.BeforeAll;
import org.junit.jupiter.api.Test;

import static org.junit.jupiter.api.Assertions.assertEquals;
import static org.mockito.BDDMockito.given;
import static org.mockito.Mockito.mock;

/**
 * EventFilterTest.java
 * TODO rely on callback integration test for now
 */
class FutureFilterTest {
    private static RpcInvocation invocation;
    private ClusterFilter eventFilter = new FutureFilter();

    @BeforeAll
    public static void setUp() {
        invocation = new RpcInvocation();
        invocation.setMethodName("echo");
        invocation.setParameterTypes(new Class<?>[] {Enum.class});
        invocation.setArguments(new Object[] {"hello"});
    }

    @Test
    void testSyncCallback() {
        @SuppressWarnings("unchecked")
        Invoker<DemoService> invoker = mock(Invoker.class);
        given(invoker.isAvailable()).willReturn(true);
        given(invoker.getInterface()).willReturn(DemoService.class);
        AppResponse result = new AppResponse();
        result.setValue("High");
        given(invoker.invoke(invocation)).willReturn(result);
        URL url = URL.valueOf("test://test:11/test?group=dubbo&version=1.1");
        given(invoker.getUrl()).willReturn(url);

        Result filterResult = eventFilter.invoke(invoker, invocation);
        assertEquals("High", filterResult.getValue());
    }

    @Test
    void testSyncCallbackHasException() throws RpcException, Throwable {
        Assertions.assertThrows(RuntimeException.class, () -> {
            @SuppressWarnings("unchecked")
            Invoker<DemoService> invoker = mock(Invoker.class);
            given(invoker.isAvailable()).willReturn(true);
            given(invoker.getInterface()).willReturn(DemoService.class);
            AppResponse result = new AppResponse();
            result.setException(new RuntimeException());
            given(invoker.invoke(invocation)).willReturn(result);
            URL url = URL.valueOf("test://test:11/test?group=dubbo&version=1.1&onthrow.method=echo");
            given(invoker.getUrl()).willReturn(url);

            eventFilter.invoke(invoker, invocation).recreate();
        });
    }
}
