/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.sidecar.utils;

import com.google.common.util.concurrent.SidecarRateLimiter;
import com.google.inject.Inject;
import com.google.inject.Singleton;
import com.google.inject.name.Named;
import io.vertx.core.AsyncResult;
import io.vertx.core.Future;
import io.vertx.core.Handler;
import io.vertx.core.Vertx;
import io.vertx.core.buffer.Buffer;
import io.vertx.core.file.CopyOptions;
import io.vertx.core.file.FileSystem;
import io.vertx.core.file.OpenOptions;
import io.vertx.core.streams.ReadStream;
import io.vertx.core.streams.WriteStream;
import java.io.File;
import java.nio.file.AtomicMoveNotSupportedException;
import org.apache.cassandra.sidecar.common.server.utils.ThrowableUtils;
import org.apache.cassandra.sidecar.utils.DigestVerifier;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@Singleton
public class SSTableUploader {
    private static final Logger LOGGER = LoggerFactory.getLogger(SSTableUploader.class);
    private static final String DEFAULT_TEMP_SUFFIX = ".tmp";
    private final FileSystem fs;
    private final SidecarRateLimiter rateLimiter;

    @Inject
    public SSTableUploader(Vertx vertx, @Named(value="IngressFileRateLimiter") SidecarRateLimiter rateLimiter) {
        this.fs = vertx.fileSystem();
        this.rateLimiter = rateLimiter;
    }

    public Future<String> uploadComponent(ReadStream<Buffer> readStream, String uploadDirectory, String componentFileName, DigestVerifier digestVerifier, String filePermissions) {
        String targetPath = StringUtils.removeEnd((String)uploadDirectory, (String)File.separator) + File.separatorChar + componentFileName;
        return this.fs.mkdirs(uploadDirectory).compose(v -> this.createTempFile(uploadDirectory, componentFileName, filePermissions)).compose(tempFilePath -> this.streamAndVerify(readStream, (String)tempFilePath, digestVerifier)).compose(verifiedTempFilePath -> this.moveAtomicallyWithFallBack((String)verifiedTempFilePath, targetPath));
    }

    private Future<String> streamAndVerify(ReadStream<Buffer> readStream, String tempFilePath, DigestVerifier digestVerifier) {
        return this.streamToFile(readStream, tempFilePath).compose(v -> digestVerifier.verify(tempFilePath)).onFailure(throwable -> this.fs.delete(tempFilePath));
    }

    private Future<Void> streamToFile(ReadStream<Buffer> readStream, String tempFilename) {
        LOGGER.debug("Uploading data to={}", (Object)tempFilename);
        return this.fs.open(tempFilename, new OpenOptions()).map(file -> new RateLimitedWriteStream(this.rateLimiter, (WriteStream<Buffer>)file)).compose(file -> {
            readStream.resume();
            return readStream.pipeTo((WriteStream)file);
        });
    }

    private Future<String> createTempFile(String uploadDirectory, String componentFileName, String permissions) {
        LOGGER.debug("Creating temp file in directory={} with name={}{}, permissions={}", new Object[]{uploadDirectory, componentFileName, DEFAULT_TEMP_SUFFIX, permissions});
        return this.fs.createTempFile(uploadDirectory, componentFileName, DEFAULT_TEMP_SUFFIX, permissions);
    }

    private Future<String> moveAtomicallyWithFallBack(String source, String target) {
        LOGGER.debug("Moving from={} to={}", (Object)source, (Object)target);
        return this.fs.move(source, target, new CopyOptions().setAtomicMove(true)).recover(cause -> {
            Exception atomicMoveNotSupportedException = (Exception)ThrowableUtils.getCause((Throwable)cause, AtomicMoveNotSupportedException.class);
            if (atomicMoveNotSupportedException != null) {
                LOGGER.warn("Failed to perform atomic move from={} to={}", new Object[]{source, target, cause});
                return this.fs.move(source, target, new CopyOptions().setAtomicMove(false));
            }
            return Future.failedFuture((Throwable)cause);
        }).compose(v -> Future.succeededFuture((Object)target));
    }

    public static class RateLimitedWriteStream
    implements WriteStream<Buffer> {
        private final SidecarRateLimiter limiter;
        private final WriteStream<Buffer> delegate;

        public RateLimitedWriteStream(SidecarRateLimiter limiter, WriteStream<Buffer> delegate) {
            this.limiter = limiter;
            this.delegate = delegate;
        }

        public WriteStream<Buffer> exceptionHandler(Handler<Throwable> handler) {
            return this.delegate.exceptionHandler(handler);
        }

        public Future<Void> write(Buffer data) {
            this.limiter.acquire(data.length());
            return this.delegate.write((Object)data);
        }

        public void write(Buffer data, Handler<AsyncResult<Void>> handler) {
            this.limiter.acquire(data.length());
            this.delegate.write((Object)data, handler);
        }

        public Future<Void> end() {
            return this.delegate.end();
        }

        public void end(Handler<AsyncResult<Void>> handler) {
            this.delegate.end(handler);
        }

        public Future<Void> end(Buffer data) {
            return this.delegate.end((Object)data);
        }

        public void end(Buffer data, Handler<AsyncResult<Void>> handler) {
            this.delegate.end((Object)data, handler);
        }

        public WriteStream<Buffer> setWriteQueueMaxSize(int maxSize) {
            return this.delegate.setWriteQueueMaxSize(maxSize);
        }

        public boolean writeQueueFull() {
            return this.delegate.writeQueueFull();
        }

        public WriteStream<Buffer> drainHandler(Handler<Void> handler) {
            return this.delegate.drainHandler(handler);
        }
    }
}

